<?php

namespace App\Http\Controllers\Api\Mobile;

use App\Http\Controllers\Controller;
use App\Http\Requests\Website\PlayListRequest;
use App\Http\Resources\MovieResource;
use App\Http\Resources\Website\PlayListResource;
use App\Models\Movie;
use App\Models\PlayList;
use App\Repositories\Interfaces\Website\PlayListRepositoryInterface;
use App\Traits\Mobile\ApiResponseTrait;
use Exception;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Log;

class PlayListController extends Controller
{
    use ApiResponseTrait;

    protected $playListRepository;

    public function __construct(PlayListRepositoryInterface $playListRepository)
    {
        $this->playListRepository = $playListRepository;
    }

    public function all()
    {
        $playlists = $this->playListRepository->all();
        return $this->responseWithSuccess(
            ['play_lists' => PlayListResource::collection($playlists)],
            __('all_playlists_retrieved_successfully')
        );
    }

    public function store(PlayListRequest $request)
    {
        try {
            $data = $request->validated();
            $playlist = $this->playListRepository->store($data);
            if (!$playlist) {
                return $this->responseWithError(
                    __('failed_to_create_playlist'),
                    400
                );
            }

            return $this->responseWithSuccess(
                $playlist,
                __('playlist_created_successfully')
            );
        } catch (Exception $e) {
            Log::error('Error creating playlist: ' . $e->getMessage(), );
            return $this->responseWithError(
                'An error occurred while creating the playlist',
                500
            );
        }
    }

    public function update(
        PlayListRequest $request,
        PlayList $playlist

    ) {
        try {
            $data = $request->validated();
            $updatedPlaylist = $this->playListRepository->update(
                $playlist,
                $data
            );

            if (!$updatedPlaylist) {
                return $this->responseWithError(
                    __('failed_to_update_playlist'),
                    400
                );
            }

            return $this->responseWithSuccess(
                $updatedPlaylist,
                __('playlist_updated_successfully')
            );
        } catch (Exception $e) {
            Log::error(__('error_updating_playlist') . $e->getMessage(), );
            return $this->responseWithError(
                'An error occurred while updating the playlist',
                500
            );
        }
    }

    public function delete(PlayList $playlist)
    {
        try {
            $deleted = $this->playListRepository->delete(
                $playlist
            );

            if (!$deleted) {
                return $this->responseWithError(
                    __('failed_to_delete_playlist'),
                    400
                );
            }

            return $this->responseWithSuccess(
                null,
                __('playlist_deleted_successfully')
            );
        } catch (Exception $e) {
            Log::error(__('error_deleting_playlist') . $e->getMessage(), );
            return $this->responseWithError(
                'An error occurred while deleting the playlist',
                500
            );
        }
    }
    public function getPlayListVideos(PlayList $playlist)
    {
        $playlistVideos = $this->playListRepository->getPlayListVideos($playlist);
        return $this->responseWithSuccess(
            [
                'playlist_details' => PlayListResource::make($this->playListRepository->find($playlist->id)),
                'play_list_videos' => MovieResource::collection($playlistVideos)
            ],
            __('all_playlist_movies_retrieved_successfully')
        );
    }
    public function addMovieToPlaylist(Request $request)
    {
        try {

            $success = $this->playListRepository->addMovieToPlaylist(
                $request->input('playlist_ids'),
                $request->input('movie_id')
            );

            if (!$success) {
                return $this->responseWithError(__('movie_could_not_be_added'), 400);
            }

            return $this->responseWithSuccess(null, __('movie_added_to_playlist'));
        } catch (Exception $e) {
            Log::error(__('error_adding_movie_to_playlist') . $e->getMessage());
            return $this->responseWithError('An error occurred while adding movie', 500);
        }
    }

    public function removeMovieFromPlaylist(Request $request)
    {
        try {
            $playlist = PlayList::find($request->input('playlist_id'));

            if (!$playlist) {
                return $this->responseWithError(__('playlist_not_found'), 404);
            }

            $success = $this->playListRepository->removeMovieFromPlaylist(
                $playlist,
                $request->input('movie_id')
            );

            if (!$success) {
                return $this->responseWithError(__('movie_could_not_be_removed'), 400);
            }

            return $this->responseWithSuccess(null, __('movie_removed_from_playlist'));
        } catch (Exception $e) {
            Log::error('Error removing movie from playlist: ' . $e->getMessage());
            return $this->responseWithError(__('error_removing_movie_from_playlist'), 500);
        }
    }
    public function getPlayListIds(Movie $movie)
    {
        return response()->json([
            'playlist_ids' => $this->playListRepository->getPlayListIds($movie),
        ]);
    }

}
