<?php

namespace App\Http\Controllers\Api\Mobile;

use App\Http\Controllers\Controller;
use App\Models\Blog;
use App\Models\BlogCategory;
use App\Models\Category;
use App\Models\Country;
use App\Models\MovieLanguage;
use App\Traits\Mobile\ApiResponseTrait;
use Illuminate\Http\Request;

class BlogController extends Controller
{
    use ApiResponseTrait;

    public function blogs()
    {
        $blogMediaFields = function ($item) {
            $item->image = apiImage(optional(json_decode($item->image))->original_image, 'blogs');
            $item->author = adminInfo('first_name') . ' ' . adminInfo('last_name');
            return $item;
        };
        $data = [
            'blogs' => Blog::where('status', 1)->paginate(10)->through($blogMediaFields),
            'filter' => [
                'categories' => Category::get(),
                'year' => range(1975, date('Y')),
            ]
        ];
        return $this->responseWithSuccess($data, __('blog_list_retrieve_successfully'));
    }

    public function blog_categories()
    {
        $categories = BlogCategory::with(['children' => function ($query) {
            $query->where('status', 1);
        }])
            ->where('status', 1)
            ->where('parent_id', 0)
            ->paginate(10);
        return $this->responseWithSuccess(['details' => $categories], __('blog_categories_list_retrieve_successfully'));
    }

    public function blog_details(Request $request)
    {
        $blogMediaFields = function ($item) {
            $item->image = apiImage(optional(json_decode($item->image))->original_image, 'blogs');
            $item->author = adminInfo('first_name') . ' ' . adminInfo('last_name');
            return $item;
        };

        $blog = Blog::where('status', 1)->where('id', $request->id)->first();

        if (!$blog) {
            return $this->responseWithError('Blog not found', 400);
        }

        $blog = $blogMediaFields($blog);

        $data = [
            'details' => $blog,
            'category' => BlogCategory::find($blog->category_id),
            'latest_blogs' => Blog::where('status', 1)->latest()->get()->transform($blogMediaFields),
        ];

        return $this->responseWithSuccess($data, __('blog_details_retrieved_successfully'));
    }


    public function blog_filter(Request $request)
    {
        $blogMediaFields = function ($item) {
            $item->image = apiImage(optional(json_decode($item->image))->original_image, 'blogs');
            $item->author = adminInfo('first_name').' '.adminInfo('last_name');
            $item->description = '';
            return $item;
        };
        $query = Blog::query();

        // Search by title (partial match)
        if ($request->filled('string')) {
            $query->where('title', 'like', '%' . $request->string . '%');
        }

        // Filter by category_id
        if ($request->filled('category_id')) {
            $query->where('category_id', $request->category_id);
        }

        // Filter by year from created_at
        if ($request->filled('year')) {
            $query->whereYear('created_at', $request->year);
        }

        $blogs = $query->latest()->paginate(10)->through($blogMediaFields);

        return $this->responseWithSuccess(['details' => $blogs], __('blog_filter_results_retrieved_successfully'));
    }
}
