<?php

namespace App\Http\Controllers\Admin;

use App\DataTables\Admin\ReviewDataTable;
use App\Http\Controllers\Controller;
use App\Http\Requests\Admin\ReviewRequest;
use App\Repositories\Interfaces\Admin\ReviewInterface;
use Exception;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Log;
use Illuminate\Support\Facades\Validator;

class ReviewController extends Controller
{
    protected $reviewRepo;

    public function __construct(ReviewInterface $reviewRepo)
    {
        $this->reviewRepo = $reviewRepo;
    }
    public function index(ReviewDataTable $dataTable)
    {
        return $dataTable->render('admin.reviews.index');
    }
    public function reviewDetails($id)
    {
        $review = $this->reviewRepo->getById($id);
        return view('admin.reviews.partials.details', compact('review'));
    }
    public function replyView($id)
    {
        $review = $this->reviewRepo->getById($id);
        return view('admin.reviews.partials.reply_modal', compact('review'));
    }
    public function reply(Request $request, $id)
    {
        if (isDemoMode()) {
            $data = [
                'status'    => 'error',
                'message'   => 'This action is disabled in demo mode.',
                'title'     => __('error'),
                'is_reload' => false,
            ];
            return response()->json($data);
        }
        $validator = Validator::make($request->all(), [
            'reply' => 'required|string|max:1000',
        ]);

        if ($validator->fails()) {
            return response()->json([
                'message' => __('validation_failed'),
                'errors' => $validator->errors()
            ], 422);
        }
        try {
            $review = $this->reviewRepo->updateReply($id, $request->reply);
            $data = [
                'status' => 'success',
                'message' => __('reply_saved_successfully'),
                'title' => __('success'),
                'is_reload' => true,
            ];
            return response()->json($data);
        } catch (Exception $e) {
            Log::error('Review reply failed', ['id' => $id, 'error' => $e->getMessage()]);
            $data = [
                'status' => 'error',
                'message' => __('failed_to_save_reply'),
                'title' => __('error'),
                'is_reload' => false,
            ];
            return response()->json($data);
        }
    }

    public function delete($id)
    {
        if (isDemoMode()) {
            $data = [
                'status'    => 'error',
                'message'   => 'This action is disabled in demo mode.',
                'title'     => __('error'),
                'is_reload' => false,
            ];
            return response()->json($data);
        }
        try {
            $this->reviewRepo->delete($id);
            return response()->json(['message' => 'Review deleted.']);
        } catch (Exception $e) {
            Log::error('Review deletion failed', ['id' => $id, 'error' => $e->getMessage()]);
            return response()->json(['message' => 'Failed to delete review.'], 500);
        }
    }
}
