<?php

namespace App\Http\Controllers\Admin;

use App\DataTables\Admin\DownloadLinkDataTable;
use App\DataTables\Admin\MovieDataTable;
use App\Http\Controllers\Controller;
use App\Http\Requests\Admin\DownloadLinkRequest;
use App\Models\DownloadLink;
use App\Repositories\Interfaces\Admin\DownloadLinkInterface;
use Exception;
use Illuminate\Http\Request;
use App\Models\Movie;
use App\Models\Episode;
use Illuminate\Support\Facades\Log;

class DownloadLinkController extends Controller
{

    public function __construct(protected DownloadLinkInterface $repository)
    {
    }
    public function index(DownloadLinkDataTable $dataTable, $type, $videoId)
    {
        $page_data['videoId'] = $videoId;
        $page_data['type'] = $type;

        return $dataTable->with([
            'videoId' => $videoId,
            'type' => $type
        ])->render('admin.download_links.index', $page_data);
    }

    public function create($type, $movie_id)
    {
        $page_data['movie_id'] = $movie_id;
        $page_data['type'] = $type;

        return view('admin.download_links.add', $page_data);
    }
    public function edit($type, $id)
    {
        $page_data['downloadLink'] = DownloadLink::findOrFail($id);
        $page_data['type'] = $type;

        return view('admin.download_links.edit', $page_data);
    }
    protected function getDownloadable(string $type, int $id)
    {
        return match ($type) {
            'movie' => Movie::findOrFail($id),
            'episode' => Episode::findOrFail($id),
            default => abort(404, 'Invalid type'),
        };
    }

    public function store(DownloadLinkRequest $request, string $type, int $movieId)
    {
        if (isDemoMode()) {
            $data = [
                'status'    => 'error',
                'message'   => 'This action is disabled in demo mode.',
                'title'     => __('error'),
                'is_reload' => false,
            ];
            return response()->json($data);
        }
        try {

            $downloadable = $this->getDownloadable($type, $movieId);
            $this->repository->create($request->all(), $downloadable);
            $data = [
                'status' => 'success',
                'message' => __('download_link_added'),
                'title' => __('success'),
                'is_reload' => true,
            ];
            return response()->json($data);

        } catch (Exception $e) {
            Log::error('Download link store failed: ' . $e->getMessage(), [
                'type' => $type,
                'movieId' => $movieId,
                'data' => $request->all()
            ]);
            $data = [
                'status' => 'error',
                'message' => __('download_link_add'),
                'title' => __('error'),
                'is_reload' => true,
            ];
            return response()->json($data);
        }
    }

    public function update(DownloadLinkRequest $request, int $id)
    {
        if (isDemoMode()) {
            $data = [
                'status'    => 'error',
                'message'   => 'This action is disabled in demo mode.',
                'title'     => __('error'),
                'is_reload' => false,
            ];
            return response()->json($data);
        }
        try {

            $this->repository->update($request->all(), $id);
            $data = [
                'status' => 'success',
                'message' => __('download_link_updated'),
                'title' => __('success'),
                'is_reload' => true,
            ];
            return response()->json($data);
        } catch (Exception $e) {
            Log::error('Download link update failed: ' . $e->getMessage(), [
                'id' => $id,
                'data' => $request->all()
            ]);
            $data = [
                'status' => 'error',
                'message' => __('download_link_update'),
                'title' => __('error'),
                'is_reload' => true,
            ];
            return response()->json($data);
        }
    }

    public function destroy(int $id)
    {
        if (isDemoMode()) {
            $data = [
                'status'    => 'error',
                'message'   => 'This action is disabled in demo mode.',
                'title'     => __('error'),
                'is_reload' => false,
            ];
            return response()->json($data);
        }

        if(!auth()->user()->can('admin.download_link.delete')){
            $data = [
                'status'    => 'error',
                'message'   => __('you_are_not_authorized_to_perform_this_action'),
                'title'     => __('error'),
                'is_reload' => false,
            ];
            return response()->json($data);
        }

        try {
            $this->repository->delete($id);
            $data = [
                'status' => 'success',
                'message' => __('download_link_deleted'),
                'title' => __('success'),
                'is_reload' => true,
            ];
            return response()->json($data);
        } catch (Exception $e) {
            Log::error('Download link delete failed: ' . $e->getMessage(), [
                'id' => $id,
            ]);
            $data = [
                'status' => 'error',
                'message' => __('download_link_delete'),
                'title' => __('error'),
                'is_reload' => true,
            ];
            return response()->json($data);
        }
    }

    public function status(Request $request)
    {
        if (isDemoMode()) {
            $data = [
                'status'    => 'error',
                'message'   => 'This action is disabled in demo mode.',
                'title'     => __('error'),
                'is_reload' => false,
            ];
            return response()->json($data);
        }
        try {
            $this->repository->statusChange($request);
            $data = [
                'status' => 'success',
                'message' => __('blog_status_update_successfully'),
                'title' => __('success'),
                'is_reload' => true,
            ];
            return response()->json($data);
        } catch (Exception $e) {
            Log::error('Download link status change failed: ' . $e->getMessage());
            $data = [
                'status' => 'error',
                'message' => __('download_link_status'),
                'title' => __('error'),
                'is_reload' => true,
            ];
            return response()->json($data);
        }
    }

}
