<?php

namespace App\DataTables\Admin;

use App\Models\MovieRequest;
use Illuminate\Database\Eloquent\Builder as QueryBuilder;
use Yajra\DataTables\Html\Builder as HtmlBuilder;
use Yajra\DataTables\Html\Column;
use Yajra\DataTables\Services\DataTable;
use Yajra\DataTables\EloquentDataTable;

class MovieRequestDataTable extends DataTable
{
    public function dataTable(QueryBuilder $query): EloquentDataTable
    {
        return (new EloquentDataTable($query))
            ->addIndexColumn()
            ->addColumn('user', function ($movieRequest) {
                $name = \Str::limit($movieRequest->name ?? '-', 50);
                $email = \Str::limit($movieRequest->email ?? '-', 50);
                return __('name') . ": $name <br>" . __('email') . ": $email";
            })
            ->addColumn('movie_name', function ($movieRequest) {
                return $movieRequest->movie_name ?? '-';
            })->addColumn('message', function ($movieRequest) {
                return \Str::limit($movieRequest->message ?? '-', 100);
            })
            ->addColumn('option', function ($movieRequest) {
                return view('admin.movie_requests.partials.action', compact('movieRequest'));
            })->rawColumns(['user'])
            ->setRowId('id');
    }

    public function query(): QueryBuilder
    {
        return MovieRequest::when($this->request->search['value'] ?? false, function ($query) {
            $search = $this->request->search['value'];
            $query->where('name', 'like', "%$search%")
                ->orWhere('email', 'like', "%$search%")
                ->orWhere('movie_name', 'like', "%$search%");
        })
            ->latest()
            ->newQuery();
    }

    public function html(): HtmlBuilder
    {
        return $this->builder()
            ->columns($this->getColumns())
            ->minifiedAjax()
            ->orderBy(1)
            ->selectStyleSingle()
            ->setTableAttribute('style', 'width:99.8%')
            ->footerCallback('function ( row, data, start, end, display ) {

                $(".dataTables_length select").addClass("form-select form-select-lg without_search mb-3");
                selectionFields();
            }')
            ->parameters([
                'dom' => 'Blfrtip',
                'buttons' => [
                    [],
                ],
                'lengthMenu' => [[10, 25, 50, 100, 250], [10, 25, 50, 100, 250]],
                'language' => [
                    'searchPlaceholder' => __('search'),
                    'lengthMenu' => '_MENU_ ' . __('movie_requests_per_page'),
                    'search' => '',
                ],
            ]);
    }

    public function getColumns(): array
    {
        return [
            Column::computed('id')->data('DT_RowIndex')->title('#')->searchable(false)->width(10),
            Column::computed('user')->title(__('user')),
            Column::computed('movie_name')->title(__('movie_name')),
            Column::computed('message')->title(__('message')),
            Column::computed('option')->title(__('option')),

        ];
    }

    protected function filename(): string
    {
        return 'Movie_Request_' . date('YmdHis');
    }


}