<?php

namespace App\DataTables\Admin;

use App\Models\Episode;
use App\Models\Season;
use App\Models\TvShow;
use Illuminate\Database\Eloquent\Builder as QueryBuilder;
use Yajra\DataTables\Html\Builder as HtmlBuilder;
use Yajra\DataTables\Html\Column;
use Yajra\DataTables\Services\DataTable;
use Yajra\DataTables\EloquentDataTable;

class EpisodeDataTable extends DataTable
{public function dataTable(QueryBuilder $query): EloquentDataTable
    {
        return (new EloquentDataTable($query))
            ->addIndexColumn()
            ->addColumn('episode_name', function ($episode) {
                return $episode->episode_name;
            })
            ->addColumn('description', function ($episode) {
                return $episode->description ?? 'N/A';
            })
            ->addColumn('source_type', function ($episode) {
                return $episode->source_type;
            })
            ->addColumn('order', function ($episode) {
                return $episode->order;
            })
            ->addColumn('season', function ($episode) {
                return Season::find($episode->season_id)->season_name ?? 'N/A';
            })
            ->addColumn('tv_series', function ($episode) {
                return TvShow::find($episode->series_id)->title ?? 'N/A';
            })
            ->addColumn('option', function ($episode) {
                return view('admin.tv_show.episode.action', compact('episode'));
            })
            ->addColumn('image', function ($episode) {
                return view('admin.tv_show.episode.image', compact('episode'));
            })
            ->setRowId('id');
    }

	public function query(): QueryBuilder
	{
        return Episode::when($this->request->search['value'] ?? false, function ($query) {
            $search = $this->request->search['value'];
            $query->where('episode_name', 'like', "%$search%");
        })
        ->latest()
        ->newQuery();
	}

	public function html(): HtmlBuilder
    {
        return $this->builder()
            ->columns($this->getColumns())
            ->minifiedAjax()
            ->orderBy(1)
            ->selectStyleSingle()
            ->setTableAttribute('style', 'width:99.8%')
            ->footerCallback('function ( row, data, start, end, display ) {

                $(".dataTables_length select").addClass("form-select form-select-lg without_search mb-3");
                selectionFields();
            }')
            ->parameters([
                'dom'        => 'Blfrtip',
                'buttons'    => [
                    [],
                ],
                'lengthMenu' => [[10, 25, 50, 100, 250], [10, 25, 50, 100, 250]],
                'language'   => [
                    'searchPlaceholder' => __('search'),
                    'lengthMenu'        => '_MENU_ '.__('episode_per_page'),
                    'search'            => '',
                ],
            ]);
    }

    public function getColumns(): array
    {
        return [
			Column::computed('id')->data('DT_RowIndex')->title('#')->searchable(false)->width(10),
            Column::computed('image')->title(__('image')),
            Column::computed('episode_name')->title(__('episode_name')),
            Column::computed('description')->title(__('description')),
            Column::computed('source_type')->title(__('source_type')),
            Column::computed('season')->title(__('season')),
            Column::computed('tv_series')->title(__('tv_series')),
            Column::computed('order')->title(__('order')),
            Column::computed('option')->title(__('option')),
           
        ];
    }

    protected function filename(): string
    {
        return 'Currency_'.date('YmdHis');
    }


}