<?php

namespace App\DataTables\Admin;

use App\Models\DownloadLink;
use App\Models\Episode;
use App\Models\Movie;
use Illuminate\Database\Eloquent\Builder as QueryBuilder;
use Yajra\DataTables\Html\Builder as HtmlBuilder;
use Yajra\DataTables\Html\Column;
use Yajra\DataTables\Services\DataTable;
use Yajra\DataTables\EloquentDataTable;

class DownloadLinkDataTable extends DataTable
{
    public function dataTable(QueryBuilder $query): EloquentDataTable
    {
        return (new EloquentDataTable($query))
            ->addIndexColumn()
            ->addColumn('title', function ($downloadLink) {
                return $downloadLink->title;
            })
            ->addColumn('link', function ($downloadLink) {
                return $downloadLink->link;
            })
            ->addColumn('resolution', function ($downloadLink) {
                return $downloadLink->resolution;
            })
            ->addColumn('file_size', function ($downloadLink) {
                return $downloadLink->file_size;
            })
            ->addColumn('status', function ($downloadLink) {
                return view('admin.download_links.status', compact('downloadLink'));
            })
            ->addColumn('option', function ($downloadLink) {
                $type = $this->type;

                return view('admin.download_links.option', compact('type', 'downloadLink'));
            })
            ->setRowId('id');
    }

    public function query(): queryBuilder
    {
        $model = match ($this->type) {
            'movie' => Movie::class,
            'episode' => Episode::class,
            default => abort(404, 'Invalid type'),
        };

        return DownloadLink::with('downloadable')
            ->where('downloadable_type', $model)
            ->where('downloadable_id', $this->videoId)
            ->latest();
    }



    public function html(): HtmlBuilder
    {
        return $this->builder()
            ->columns($this->getColumns())
            ->minifiedAjax()
            ->orderBy(1)
            ->selectStyleSingle()
            ->setTableAttribute('style', 'width:99.8%')
            ->footerCallback('function ( row, data, start, end, display ) {

                $(".dataTables_length select").addClass("form-select form-select-lg without_search mb-3");
                selectionFields();
            }')
            ->parameters([
                'dom' => 'Blfrtip',
                'buttons' => [
                    [],
                ],
                'lengthMenu' => [[10, 25, 50, 100, 250], [10, 25, 50, 100, 250]],
                'language' => [
                    'searchPlaceholder' => __('search'),
                    'lengthMenu' => '_MENU_ ' . __('movie_files_per_page'),
                    'search' => '',
                ],
            ]);
    }

    public function getColumns(): array
    {
        return [
            Column::computed('id')->data('DT_RowIndex')->title('#')->searchable(false)->width(10),
            Column::computed('title')->title(__('title')),
            Column::computed('resolution')->title(__('resolution')),
            Column::computed('file_size')->title(__('file_size')),
            Column::computed('link')->title(__('link')),
            Column::computed('status')->title(__('status')),
            Column::computed('option')->title(__('option')),

        ];
    }

    protected function filename(): string
    {
        return 'Currency_' . date('YmdHis');
    }


}